# to autoload this file
proc m2Edit.tcl {} {}

#####################################################################################
# 
#   Author    Date        Modification
#   ------    ----        ------------
#    af       21.02.00    Replacing braces in all regexp statements like 
#                          regexp {[ \t]*PROCEDURE[ \t]+([^\s;(]+)} $t all procName
# 			  with quotes and preceeding brackets "[" and "]" with "\" like
# 			   regexp "\[ \t\]*PROCEDURE\[ \t\]+(\[^\\r;(\]+)" $t all procName
#                         whenever a "\t" is in the expression (hint by 
#                         Dominique Dhumier).  Since I'm not sure how well the
#                         new expressions really are, I have left the previous ones
#                         as comments before the new ones.



#================================================================================
#  M2 Editing (also in templates) and Moving around in a file  #
#================================================================================

namespace eval M2 {}


#================================================================================
proc getCurLine {} {
    set pos [getPos]
    set start [lineStart $pos]
    set end [nextLineStart $pos]
    set text [getText $start $end]
    regexp "(\[^\r\]*)\r?" $text dummyText text
    return $text
}

#================================================================================
proc isNotMODModule {} {
    # This won't work in Alpha 8
    set firstWord [lindex [getText [minPos] [nextLineStart [minPos]]] 0]
    if {("$firstWord" == "MODULE") | ("$firstWord" == "IMPLEMENTATION")} then {
	return 0
    } else {
	return 1
    }
}


#================================================================================
# discardBullet performs a mode specific return, i.e. jumps out of the current line 
# (does NOT really break it, see proc breakTheLine for that other behavior) and does 
# insert a new line which is indented according to context.  The latter means, that 
# indentation is done in a Modula-2 language specific way, either new line is 
# indented the same way as the previous line, e.g.
#
#          x := 7*y;
#          |<- hitting return makes new line and moves cursor here 
#              if cursor has been anywhere within above line
# 
# or is indented by one level more if the scope changes, e.g.
#
#     IF (a<=b) THEN
#       |<- hitting return makes new line and moves cursor here 
#           if cursor has been anywhere within above line
#     ELSE
#     END(*IF*);
#
# Note, uses procs jumpOutOfLnAndReturn and modulaTab. 

proc discardBullet {} {
    set oldPos [getPos]
    set matchStr "\[ \\t\]*"
    set end [lindex [search -s -r 1 -f 1  -i 0 -n -- $matchStr $oldPos] 1]
    if {$end != ""} then {
	set newPos [pos::math $end - 1]
	select $newPos $end
	# set text [getText $newPos $end]
	# alertnote "pos $pos, end $end, text = '$text'"
	# alertnote "in discardBullet"
	if { [isSelection] } { deleteSelection }
	if {[pos::compare $oldPos == $newPos]} then {
	    set moved 0
	} else {
	    set moved 1
	}
	# alertnote "moved = $moved"
	return $moved
    } else {
	return 0
    }
}

proc jumpToTemplatePlaceHolder {} {
    global returnWords
    global returnCompleteWords
    global posBeforeJumpOutOfLn
    global selEndBeforeJumpOutOfLn
    # returns 0 nothing more needs to be done by callee, was just a jump
    #         1 just an additional action by callee necessary (no indentation)
    #         2 same as 1 but plus indentation
    #         3 similar to 1 but no language specific handling could be determined
    set posBeforeJumpOutOfLn [getPos]
    set selEndBeforeJumpOutOfLn [selEnd]
    set line [getCurLine]
    set first [firstWord $line]
    set first [trim $first]
    if {[lsearch " $returnWords " $first] > -1} {
	# reserved word $first found
	if {[lsearch " $returnCompleteWords " $first] > -1} {
	    set pos [getPos]
	    set start [lineStart $pos]
	    set leftText [getText $start $pos]
	    if {$first == "IF"} {
		if { [regexp {^(.+)( ELSE )(.+)$} $line] } then {
		    if {[string first "ELSE" $leftText] > -1} {
			# jumpOutOfLnAndReturn
			return 1
		    }
		    if {[string first "THEN" $leftText] > -1} {
			goto [pos::math $start + [string first "ELSE" $line] + 5]
			discardBullet
			return 0
		    }
		    goto [pos::math $start + [string first "THEN" $line] + 5]
		    discardBullet
		    return 0
		} elseif {[regexp {END} $line]} then {
		    return 1
		} else {
		    # its an "ordinary" IF spread over several lines (not on one single line)
		    # jumpOutOfLnAndReturn
		    # modulaTab
		    return 2
		}
	    }
	    if {$first == "FOR"} {
		if {[regexp {END} $line]} then {
		    return 1
		}
		if {[string first "BY" $leftText] > -1} {
		    # jumpOutOfLnAndReturn
		    # modulaTab
		    return 2
		}
		if {[string first "TO" $leftText] > -1} {
		    if { [regexp {^(.+)( BY )(.+)$} $line] } then {
			goto [pos::math $start + [string first "BY" $line] + 3]
			discardBullet
			return 0
		    } elseif {[regexp {END} $line]} then {
			return 1
		    } else {
			# jumpOutOfLnAndReturn
			# modulaTab
			return 2
		    }
		}
		if {[string first ":=" $leftText] > -1} {
		    goto [pos::math $start + [string first "TO" $line] + 3]
		    discardBullet
		    return 0
		}
		if {[string first "FOR" $leftText] > -1} {
		    goto [pos::math $start + [string first ":=" $line] + 3]
		    discardBullet
		    return 0
		}
		goto [pos::math $start + [string first "FOR" $line] + 4]
		discardBullet
		return 0
	    }
	    if {$first == "FROM"} {
		if {[string first "IMPORT" $leftText] > -1} {
		    # jumpOutOfLnAndReturn
		    return 1
		}
		if {[string first "FROM" $leftText] > -1} {
		    goto [pos::math $start + [string first "IMPORT" $line] + 7]
		    discardBullet
		    return 0
		}
		goto [pos::math $start + [string first "FROM" $line] + 5]
	    }
	    discardBullet
	    return 0
	} else {
	    # jumpOutOfLnAndReturn
	    # modulaTab
	    if {($first == "PROCEDURE") && [isNotMODModule]} then {
		return 1
	    } elseif {[regexp {END} $line]} then {
		return 1
	    } else {
		return 2
	    }
	}
    } else {
	# no reserved word found
	# jumpOutOfLnAndReturn
	return 3
    }
}

# Alternative to Tab (used to be plain Return)
proc tabOrJumpOutOfLnAndReturn {} {
    global M2RightShift
    set result [jumpToTemplatePlaceHolder]
    if !$result {return} else {
	switch $result {
	    1      { jumpEOLNewLnIndent "" }
	    2      { jumpEOLNewLnIndent $M2RightShift }
	    default { jumpEOLNewLnIndent "" }
	}
    }	
}

# Tab handles auto place holders in template supported Modula-2 constructs like Return 
proc tabKey {} {
    global M2RightShift
    set result [jumpToTemplatePlaceHolder]
    if !$result {return} else {
	switch $result {
	    1      { if {![discardBullet]} then {jumpEOLNewLnIndent "" } }
	    2      { if {![discardBullet]} then {jumpEOLNewLnIndent $M2RightShift} }
	    default { if {![discardBullet]} then {modulaTab} }
	}
    }	
}

#========================== Begin of Alpha Pre7 routines =================================
# The following routines are only kept here to support pre version 7 Alpha, like 6.5.2, in
# an upward compatible manner of M2 mode. modulaReturn is bound to key carriage return. 
# Under Alpha >= 7 these routines should never be called. See tabOrJumpOutOfLnAndReturn instead.
proc modulaReturn {} {
    global returnWords
    global returnCompleteWords
    set line [getCurLine]
    set first [firstWord $line]
    set first [trim $first]
    if {[lsearch " $returnWords " $first] > -1} {
	if {[lsearch " $returnCompleteWords " $first] > -1} {
	    set pos [getPos]
	    set start [lineStart $pos]
	    set leftText [getText $start $pos]
	    if {$first == "FOR"} {
		if {[string first "TO" $leftText] > -1} {
		    indentOnReturn
		    return
		}
		if {[string first ":=" $leftText] > -1} {
		    goto [pos::math $start + [string first "TO" $line] + 3]
		    return
		}
		if {[string first "FOR" $leftText] > -1} {
		    goto [pos::math $start + [string first ":=" $line] + 3]
		    return
		}
		goto [pos::math $start + [string first "FOR" $line] + 4]
	    }
	    if {$first == "FROM"} {
		if {[string first "IMPORT" $leftText] > -1} {
		    actionOnReturn
		    return
		}
		if {[string first "FROM" $leftText] > -1} {
		    goto [pos::math $start + [string first "IMPORT" $line] + 7]
		    return
		}
		goto [pos::math $start + [string first "FROM" $line] + 5]
	    }
	} else {
	    indentOnReturn
	}
    } else {
	actionOnReturn
    }
}

proc indentOnReturn {} {
    global M2RightShift
    actionOnReturn
    insertText $M2RightShift
}

proc actionOnReturn {} {
    set pos [getPos]
    deleteText $pos [selEnd]
    goto $pos
    endOfLine
    carriageReturn	
}

#================================= End of Alpha Pre7 routines ===================

 
#================================================================================
# returns the amount of white space needed from the begin of the line as defined
# by the first, non-white space line preceeding the line which contains pos. 
# This algorithm assumes that there are no tab (ASCII ht=11C) characters within 
# the text. Preserves current position.
proc getIndentation {pos} {
    set curPos [getPos]
    set curLineBeg [lineStart $pos]
    set nextLineBeg [nextLineStart $pos]
    if {[pos::compare $curLineBeg != $nextLineBeg]} then {
	# it is not just an empty line at the end of the file
	# omit the carriage return at the end of the string
	set nextLineBeg [pos::math [nextLineStart $pos] - 1]
    }
    select $curLineBeg $nextLineBeg
    tabsToSpaces
    set text [getText $curLineBeg $nextLineBeg ] 
    # set text [getText $curLineBeg [expr [nextLineStart $pos] - 1] ]
    # regexp {(^[ \t]*)(.*)$} $text all theIndentation rest
    regexp "(^\[ \t\]*)(.*)$" $text all theIndentation rest
    # alertnote "curLineBeg = $curLineBeg, theIndentation = '$theIndentation', rest = '$rest'"
    while {($rest == "") && ([pos::compare $curLineBeg != [minPos]])} {
	# line was empty or just white space, search above
	previousLine
	set pos [getPos]
	# alertnote "in getIndentation pos = $pos"
	set otherLineBeg [lineStart $pos]
	set nextLineBeg [pos::math [nextLineStart $pos] - 1]
	select $otherLineBeg $nextLineBeg
	tabsToSpaces
	set text [getText $otherLineBeg $nextLineBeg ]
	# regexp {(^[ \t]*)(.*)$} $text all theIndentation rest
	regexp "(^\[ \t\]*)(.*)$" $text all theIndentation rest
    }
    goto $curPos
    return $theIndentation
}

#================================================================================
# Moves the cursor to a new line next, next to the current one and indents 
# the insertion (cursor) by inserting white space with what's returned by 
# getIndentation.
proc jumpOutOfLnAndReturn {} {
    global M2RightShift
    global posBeforeJumpOutOfLn
    global selEndBeforeJumpOutOfLn
    set posBeforeJumpOutOfLn [getPos]
    set selEndBeforeJumpOutOfLn [selEnd]
    if { [isSelection] } { deleteSelection }
    set whiteSpace [getIndentation [getPos]]
    set result [jumpToTemplatePlaceHolder]
    endOfLine
    if {$result == 2} then { set whiteSpace "$whiteSpace$M2RightShift" }
    insertText "\r$whiteSpace"
}

#================================================================================
# Very similar to jumpOutOfLnAndReturn, but does no $M2RightShift.
proc jumpOutOfLnAndRet {} {
    global M2RightShift
    global posBeforeJumpOutOfLn
    global selEndBeforeJumpOutOfLn
    set posBeforeJumpOutOfLn [getPos]
    set selEndBeforeJumpOutOfLn [selEnd]
    if { [isSelection] } { deleteSelection }
    set whiteSpace [getIndentation [getPos]]
    endOfLine
    insertText "\r$whiteSpace"
}


#================================================================================
# Similar to jumpOutOfLnAndReturn but no scope or other language specific context 
# analyzed; on the other hand it accepts some identation amount to be added (allows
# for single Cmd^Z undo). Basic routine needed by many other scripts or template 
# procedures.  Needs getIndentation.
proc jumpEOLNewLnIndent {indentation} {
	if { [isSelection] } { deleteSelection }
	set whiteSpace "[getIndentation [getPos]]$indentation"
	endOfLine
	insertText "\r$whiteSpace"
}

#================================================================================
# M2 Carriage return but skips first next line. This is convenient, e.g. in
# following situaition
# 
#  PROCEDURE MyProc;
#  BEGIN
#    |<-   here is the cursor now
#  END MyProc;
#  |<-   here you would like to jump to continue typing
# 
#  use skipLnReturn (bound to CMD-RETURN) to get the desired result.
proc skipLnReturn {} {
	global posBeforeJumpOutOfLn
	global selEndBeforeJumpOutOfLn
	set curPos [getPos]
	set curSelEnd [selEnd]
	if { [isSelection] } { deleteSelection }
	endOfLine
	forwardChar
	jumpOutOfLnAndReturn
	set posBeforeJumpOutOfLn $curPos
	set selEndBeforeJumpOutOfLn $curSelEnd
}

proc openNewAbove {} {
	global posBeforeJumpOutOfLn
	global selEndBeforeJumpOutOfLn
	set curPos [getPos]
	set curSelEnd [selEnd]
	if { [isSelection] } { deleteSelection }
	beginningOfLine
	backwardChar
	jumpOutOfLnAndReturn
	set posBeforeJumpOutOfLn $curPos
	set selEndBeforeJumpOutOfLn $curSelEnd
}

proc skipPrevLnOpenNew {} {
	global posBeforeJumpOutOfLn
	global selEndBeforeJumpOutOfLn
	set curPos [getPos]
	set curSelEnd [selEnd]
	if { [isSelection] } { deleteSelection }
	beginningOfLine
	backwardChar
	beginningOfLine
	backwardChar
	jumpOutOfLnAndReturn
	set posBeforeJumpOutOfLn $curPos
	set selEndBeforeJumpOutOfLn $curSelEnd
}
	


#================================================================================
# Combines effect of breakTheLine and indentCurLine, was breakLineAndIndent
proc M2::carriageReturn {} {
    global M2RightShift
    if { [isSelection] } { deleteSelection }
    set curPos [getPos]
    set result [jumpToTemplatePlaceHolder]
    # the following would allow to jump within line
    # if !$result {return}
    set whiteSpace [getIndentation $curPos]
    if {$result == 2} then { insertText "\r$whiteSpace$M2RightShift" } else { insertText "\r$whiteSpace" }
}


#================================================================================
# Breaks the line at the current position (old fashioned plain-vanilla return 
# without any indentation)
proc breakTheLine {} {
    if { [isSelection] } { deleteSelection }
    insertText "\r"
}

#================================================================================
# Moves insertion point to the indentation position (as returned by getIndentation)
# within same line
# Note, is based on getIndentation. 
proc indentCurLine {} {
    set beg [lineStart [getPos]]
    previousLine
    set whiteSpace [getIndentation [getPos]]
    goto $beg
    insertText "$whiteSpace"
}

#================================================================================
# In contrast to indentCurLine does also really indent one level further
proc M2::indentLine {} {
    indentCurLine
    modulaTab
}

#================================================================================
# Used by smartPaste feature (package) which unfortunately assumes, this procedure
# is only called if the current line contains only whitespace (?)
proc M2::correctIndentation {pos {next ""}} {
    global M2RightShift
    set curPos [getPos]
    set whiteSpace [getIndentation $pos]
    # assume this routine is never called if current line contains something <> white-space
    previousLine
    # make sure not to leave current line by backwardWord in case it contains only a single word
    endOfLine
    backwardWord
    set result [jumpToTemplatePlaceHolder]
    if {$result != 0} then { goto $curPos }
    if {$result == 2} then { set whiteSpace "$whiteSpace$M2RightShift" }
    # alertnote "result = $result / length(whiteSpace) = [string length $whiteSpace]"
    # maybe the following helps copyRing to collaborate better with smartPaste?
    insertText $whiteSpace
    return [string length $whiteSpace]
}


#================================================================================
# Moves current line such that its first non white-space char is indented
# the same as the previous line
proc adjCurLnToIndentAbove {} {
    set oldPos [getPos]
    set beg [lineStart [getPos]]
    previousLine
    set whiteSpace [getIndentation [getPos]]
    goto $beg
    set oldPos [getPos]
    oneSpace
    set newPos [getPos]
    if {[pos::compare $oldPos != $newPos]} then {
	# oneSpace has changed position since there was actually
	# some white space to the right of the current position
	backSpace
    }
    insertText "$whiteSpace"
}

#================================================================================
# Moves current line such that its first non white-space char is indented
# the same as the next non-white space line
proc adjCurLnToIndentBelow {} {
    set beg [lineStart [getPos]]
    endOfLine
    forwardWord
    set whiteSpace [getIndentation [getPos]]
    goto $beg
    set oldPos [getPos]
    oneSpace
    set newPos [getPos]
    if {[pos::compare $oldPos != $newPos]} then {
	# oneSpace has changed position since there was actually
	# some white space to the right of the current position
	backSpace
    }
    insertText "$whiteSpace"
}


#================================================================================
proc unIndent {} {
    global M2RightShift
    set count [string length $M2RightShift]
    set beg [lineStart [getPos]]
    for {set i 0} {$i < $count} {incr i} {
	if {[pos::compare [getPos] > $beg]} then {backwardCharSelect}
    }
    if { [isSelection] } { deleteSelection }
}

#================================================================================
# Indents insertion point from current position by M2RightShift
proc modulaTab {} {
	global M2RightShift
	insertText $M2RightShift
}


#================================================================================
proc resumeBeforeCarRet {} {
    global posBeforeJumpOutOfLn
    global selEndBeforeJumpOutOfLn
    if {[info exists posBeforeJumpOutOfLn] && [info exists selEndBeforeJumpOutOfLn]} then {
	select $posBeforeJumpOutOfLn $selEndBeforeJumpOutOfLn
    }
}

# Initialize posBeforeJumpOutOfLn and selEndBeforeJumpOutOfLn
set posBeforeJumpOutOfLn [minPos]
set selEndBeforeJumpOutOfLn [minPos]


#================================================================================
proc newLnAtSameCol {} {
    if { [isSelection] } { deleteSelection }
    set pos [getPos]
    set col [pos::diff $pos [lineStart $pos]]
    endOfLine
    set whiteSpace ""
    for {set i 0} {$i < $col} {incr i} {
	set whiteSpace "$whiteSpace "
    }
    insertText "\r$whiteSpace"
}


#================================================================================
# Does completions of templates if preceeding reserved word is matched. Triggered
# with the space you typically type after a reserved word (I think much more
# comfortable than Alpha's conventional completion (but, BTW, which does also work 
# the conventional way).
proc modulaSpace {} {
    global spaceWords
    set line [getCurLine]
    set first [firstWord $line]
    set first [trim $first]
    set rest [restWord $line]
    set rest [trim $rest]
    if {[lsearch " $spaceWords " $first] > -1} {
	if {[string length $rest] > 0} {
	    deleteText [getPos] [selEnd]
	    insertText " "
	} else {
	    if {[catch template$first]} {
		beep
		alertnote "Template for:$first not defined"
	    } 			
	}
    } else {
	deleteText [getPos] [selEnd]
	insertText " "
    }
}

#================================================================================
# The actual completion procedure of M2 mode.  Kept here for upward compatibility
# and support of the traditional M2 menu. The latter offers also most templates, 
# which offer more comfort than just a completion (file, window handling etc. all 
# accessible with a single menu command!). IMPLEMENTATION NOTE: If this routine
# can't expand itself it will call the predefined completion routine, making
# words already in existence available for completion (the usual behavior 
# expected by Alpha users).
proc expandSpace {} {
    global expandWords
    global doubleDefinedWords
    set pos [getPos]
    backwardWord
    set bPos [getPos]
    if {[pos::compare $bPos == [pos::math [minPos] + 1]]} {
	set text " "
	regexp "\[A-Za-z\]" [lookAt [minPos]] text
	if {$text != " "} {
	    set bPos [minPos]
	}
    }
    
    forwardWord
    set fPos [getPos]
    goto $pos
    set origWord [getText $bPos $fPos]
    set word [string toupper $origWord]
    set ind [lsearch $expandWords $origWord*]
    set thereIsCompletion [lsearch $doubleDefinedWords $origWord*]
    if {($ind == -1) | ($thereIsCompletion != -1)} {
	# wordCompletion does no longer work in Alpha > 7.x 
	bind::Completion
	return
    }
    set expandWord [lindex $expandWords $ind]
    if {$expandWord != $origWord} {
	replaceText $bPos $fPos $expandWord
    } 
}

#================================================================================
proc completePrevWord {} {
	set pos [getPos]
	backwardWord
	forwardWord
	bind::Completion
}

#================================================================================
proc killWholeLine {} {
	goto [lineStart [getPos]]
	killLine
	KillLnIfEmpty
}

#================================================================================
proc KillPrevLnIfOnlyComment {} {
    previousLine
    set lnBeg [lineStart [getPos]]
    set lnEnd [pos::math [nextLineStart [getPos]] -1]
    set prevLnTxt [getText $lnBeg $lnEnd]
    # regexp {^[ \t]*\(\*.*\*\)[ \t]*$} $prevLnTxt
    if {[regexp "^\[ \t\]*\\(\\*.*\\*\\)\[ \t\]*$" $prevLnTxt ]} then {
	# line contains really just a comment, delete it
	killLine
	killLine
	return 1
    } else {
	# not just a comment, resume position
	nextLine
	return 0
    }
}

#================================================================================
proc KillNextLnIfOnlyComment {} {
    nextLine
    set lnBeg [lineStart [getPos]]
    set lnEnd [pos::math [nextLineStart [getPos]] -1]
    set nextLnTxt [getText $lnBeg $lnEnd]
    # regexp {^[ \t]*\(\*.*\*\)[ \t]*$} $nextLnTxt
    if {[regexp "^\[ \t\]*\\(\\*.*\\*\\)\[ \t\]*$" $nextLnTxt ]} then {
	# line contains really just a comment, delete it
	previousLine
	killLine
	killLine
	return 1
    } else {
	# not just a comment, resume position
	previousLine
	return 0
    }
}


#================================================================================
proc KillLnIfEmpty {} {
    set savePos [getPos]
    set lnBeg [lineStart [getPos]]
    set lnEnd [pos::math [nextLineStart [getPos]] -1]
    # beginningLineSelect appears to be buggy
    # endLineSelect
    select $lnBeg $lnEnd
    set theLine [getSelect]
    # regexp {^[ \t]*$} $theLine
    if {[regexp "^\[ \t\]*$" $theLine]} then {
	# is really empty line
	clear ; killLine 
	return 1
    } else {
	# not empty line
	goto $savePos
	return 0
    }
}

#================================================================================
proc JoinToOneSpace {} {
	endOfLine 
	deleteChar
	oneSpace
}


#================================================================================
proc notAComment {} {
    set pos [getPos]
    set end [selEnd]
    set comBeg [getText $pos [pos::math $pos + 2]]
    set comEnd [getText [pos::math $end -2] $end]
    # alertnote "pos = $pos / end = $end / comBeg = '$comBeg' / comEnd = '$comEnd'"
    if {($comBeg == "(*") && ($comEnd == "*)")} then {
	# alertnote "notAComment = 0"
	return 0
    } else {
	if {([pos::compare $pos == $end]) || ([pos::compare $pos <= [minPos]]) || ([pos::compare $end >= [maxPos]])} then {
	    # abort if end reached
	    # alertnote "notAComment = 0"
	    return 0
	} else {
	    # alertnote "notAComment = 1"
	    return 1
	}
    }
}


#================================================================================
proc selectM2Comment {} {
    balance
    set pos [getPos]
    set end [selEnd]
    if {[pos::compare $pos == $end]} {
	# first balance failed
	message "Cursor not in comment, missing '(*', '*)', or non-paired (),{},\[\]"
	return {0 1}
    }
    while {([pos::compare $pos > [minPos]]) && ([pos::compare $end < [maxPos]]) && [notAComment]} {
	balance
	set pos [getPos]
	set end [selEnd]
    }
    if {[pos::compare $pos == $end]} {
	message "Couldn't find begin or end of comment, or it contains non-paired parantheses"
	return {0 0}
    } else {	
	message "Comment selected"
	return {1 0}
    }
}

#================================================================================
proc selectNestedM2Comment {} {
    set comBeg [getPos]
    set comEnd [getPos]
    balance
    set pos [getPos]
    set end [selEnd]
    if {[pos::compare $pos == $end]} {
	# first balance failed
	message ""  # clears Abort message
	return {0 1}
    }
    while {([pos::compare $pos > [minPos]]) && ([pos::compare $end < [maxPos]])} {
	while {([pos::compare $pos != $end]) && [notAComment]} {
	    balance
	    set pos [getPos]
	    set end [selEnd]
	}
	if {[pos::compare $pos == $end]} {
	    # balance failed
	    message ""  # clears Abort message
	    break
	} else {	
	    # is a comment
	    set comBeg $pos
	    set comEnd $end
	    # try one more
	    balance
	    set pos [getPos]
	    set end [selEnd]
	}
    }
    # now select last comment (if any)
    select $comBeg $comEnd
    if {([pos::compare $comBeg != $comEnd])} then {
	message "Nested comment selected"
	return {1 0}
    } else {
	message ""  # clears Abort message
	return {0 0}
    }
}


#================================================================================
proc wrapComment {} {
    global leftFillColumn
    global M2RightShift
    global M2WrapRightMargin
    global fillColumn
    global curAlphaV
    set increment [string length $M2RightShift]
    set selComment [selectM2Comment]
    set succeeded [lindex $selComment 0]
    set firstFailed [lindex $selComment 1]
    if !$succeeded {
	beep
	if $firstFailed {
	    message "Cursor not in comment, missing '(*', '*)', or non-paired (),{},\[\]"
	} else {
	    message "Couldn't find begin or end of comment, or it contains non-paired parantheses"
	}
	return
    }
    set firstPos [getPos]
    set lastPos [selEnd]
    # alertnote "pos = $pos / end = $end / firstPos = $firstPos / lastPos = $lastPos"
    goto [pos::math $firstPos + 2]
    # jumpOutOfLnAndReturn
    insertText "\r"
    indentCurLine
    set lastPos [matchIt "\(" [pos::math $firstPos +$increment]]
    set pos [getPos]
    set end [pos::math $lastPos +1]
    select $pos $end
    tabsToSpaces
    set tmpLeftFillColumn $leftFillColumn
    set tmpfillColumn $fillColumn
    set leftMargColumn [expr [lindex [posToRowCol $firstPos] 1] + $increment]
    if {$curAlphaV <= "6.01"} then {
	# Has old wrap behavior, which requires following statement
	set leftFillColumn $leftMargColumn
	set fillColumn $M2WrapRightMargin
    } else {
	# Has new wrap behavior
	set leftFillColumn $increment
	set fillColumn [expr $M2WrapRightMargin -$leftMargColumn]
    }
    fillRegion
    set leftFillColumn $tmpLeftFillColumn
    set fillColumn $tmpfillColumn
    goto [pos::math [matchIt "\(" [pos::math $firstPos +$increment]] -1]
    insertText "\r"
    indentCurLine
    set curPos [getPos]
    goto [pos::math $curPos -2]
    unIndent
    set topTxtLeftMargRow [lindex [posToRowCol $firstPos] 0]
    set topTxtLeftMargRow [expr $topTxtLeftMargRow +1]
    set topTxtLeftMarg [rowColToPos $topTxtLeftMargRow 0]
    set textBeg [expr [lindex [posToRowCol $firstPos] 1] + $increment]
    set count [expr $textBeg]
    goto $topTxtLeftMarg
    if {$curAlphaV <= "6.01"} then {
	# Has old wrap behavior, which requires following statement
	for {set i 0} {$i < $count} {incr i} {
	    deleteChar
	}
    } else {
	for {set i 0} {$i < $increment} {incr i} {
	    deleteChar
	}
    }
    goto [pos::math $firstPos + 2]
    balance
    set end [selEnd]
    # delete extra line which was inserted by algorithm
    select $end [pos::math $end + 1]
    deleteSelection
    # to to begin of last line
    previousLine
    # Sometimes empty line between text and *) is created, cursors sits in it => clear it
    KillLnIfEmpty
    message "Comment wrapped"
}


#================================================================================
proc wrapText {} {
    global leftFillColumn
    global fillColumn
    global M2WrapRightMargin
    global curAlphaV
    set pos [getPos]
    set end [selEnd]
    if {[pos::compare $pos == $end]} {
	beep
	message "Please make a selection"
	return
    }
    set firstPos [search -s -r 1 -f 1 -n -- "\[\^ \\t\\r\]" $pos]
    set firstPosBeg [lindex $firstPos 0]
    if {[pos::compare $firstPosBeg > $end]} {
	beep
	message "No text in selection"
	return
    }
    tabsToSpaces
    set tmpLeftFillColumn $leftFillColumn
    set tmpfillColumn $fillColumn
    set leftMargColumn [lindex [posToRowCol $firstPos] 1]
    if {$curAlphaV <= "6.01"} then {
	# Has old wrap behavior, which requires following statement
	set leftFillColumn $leftMargColumn
	set fillColumn $M2WrapRightMargin
    } else {
	# Has new wrap behavior
	set fillColumn [expr $M2WrapRightMargin -$leftMargColumn]
    }
    fillRegion
    set leftFillColumn $tmpLeftFillColumn
    set fillColumn $tmpfillColumn
    
    set topTxtLeftMargRow [lindex [posToRowCol $firstPos] 0]
    set topTxtLeftMarg [rowColToPos $topTxtLeftMargRow 0]
    set textBeg [lindex [posToRowCol $firstPos] 1]
    set count [expr $textBeg]
    goto $topTxtLeftMarg
    if {$curAlphaV <= "6.01"} then {
	# Has old wrap behavior, which requires following statement
	for {set i 0} {$i < $count} {incr i} {
	    deleteChar
	}
    }
    goto $pos
    message "Text wrapped"
}



#================================================================================
proc encloseSelection {} {
    global M2modeVars
    set pos [getPos]
    set end [selEnd]
    if {[pos::compare $pos == $end]} {
	beep
	message "Please make a selection"
	return
    }
    replaceText $pos $end "$M2modeVars(prefixString)[getText $pos $end]$M2modeVars(suffixString)"
    select $pos [pos::math $end + [string length $M2modeVars(prefixString)] + [string length $M2modeVars(suffixString)]]
}

#================================================================================
proc unencloseSelection {} {
    global M2modeVars
    set pos [getPos]
    set end [selEnd]
    if {[pos::compare $pos == $end]} {
	beep
	message "Please make a selection"
	return
    }
    set prefixLe [string length $M2modeVars(prefixString)]
    set suffixLe [string length $M2modeVars(suffixString)]
    if {[getText $pos [pos::math $pos + $prefixLe]] != "$M2modeVars(prefixString)"} {
	beep
	message "Begin of selection different from '$M2modeVars(prefixString)'"
	return
    }
    if {[getText [pos::math $end - $suffixLe] $end] != "$M2modeVars(suffixString)"} {
	beep
	message "End of selection different from '$M2modeVars(suffixString)'"
	return
    }
    replaceText [pos::math $end - $suffixLe] $end ""
    replaceText $pos [pos::math $pos + $prefixLe] ""
    select $pos [pos::math $end - $suffixLe - $prefixLe]
}

#================================================================================
proc commentSelection {} {
    set pos [getPos]
    set end [selEnd]
    if {[pos::compare $pos == $end]} {
	beep
	message "Please make a selection"
	return
    }
    if {[getText [pos::math $end - 1] $end] == "\r"} {
	# selection end is at begin of line => don't insert blank
	replaceText $pos $end "\(\*\. [getText $pos $end]\.\*\)"
	set addedChars 7
    } else {
	replaceText $pos $end "\(\*\. [getText $pos $end] \.\*\)"
	set addedChars 8
    }
    select $pos [pos::math $end + $addedChars]
}

#================================================================================
proc uncommentSelection {} {
    set pos [getPos]
    set end [selEnd]
    if {[pos::compare $pos == $end]} {
	beep
	message "Please make a selection"
	return
    }
    if {[pos::diff $end $pos] < 8} {
	beep
	message "Selection to small"
	return
    }
    set leftSize 4 
    if {[getText $pos [pos::math $pos + $leftSize]] != "(*. "} {
	set leftSize 3
	if {[getText $pos [pos::math $pos + $leftSize]] != "(*."} {
	    beep
	    message "Wrong left comment-start in selection"
	    return
	}
    }	
    set rightSize 4
    if {[getText [pos::math $end - $rightSize] $end] != " .*)"} {
	set rightSize 3
	if {[getText [pos::math $end - $rightSize] $end] != ".*)"} {
	    beep
	    message "Wrong right comment-start in selection"
	    return
	}
    }
    replaceText [pos::math $end - $rightSize] $end ""
    replaceText $pos [pos::math $pos + $leftSize] ""
    select $pos [pos::math $end - $leftSize - $rightSize]
}


 
#================================================================================
# This procedure is a replacement for textManip.tcl uncommentLine, which
# does not properly reverse the effect of Alpha's menu command 
# "Text > Comment Line" (Cmd^D), since it does not call removeSuffix
proc M2::uncommentLine {} { 
	removePrefix; removeSuffix; 
	# deselect
	select [getPos] [getPos]
	# message "Line uncommented"
}

#================================================================================
# This procedure is from Juan Falgueras, thanks!
proc myWrapObject {left right} {
	set currentPos [getPos]
# 	set selected [isSelection]
	if {[isSelection]} then {
		replaceText $currentPos [selEnd] $left [getSelect] "$right"
	} else {
		insertText $left "" "$right"
	}
	goto $currentPos
# 	ring::+
	tabKey
# 	return $selected
}

#================================================================================
proc doM2ShiftLeft {} {
    global M2LeftShift
    if {![isSelection]} then {selectLine}
    set start [lineStart [getPos]]
    set end   [nextLineStart [pos::math [selEnd] -1]]
    select $start $end
    tabsToSpaces
    set increment [string length $M2LeftShift]
    for {set i $start} {[pos::compare $i < $end]} {set i [nextLineStart $i]} {
	if {[getText $i [pos::math $i + $increment]] != $M2LeftShift} {
	    beep
	    return
	} 
    }
    select $start $start
    for {set i $start} {[pos::compare $i < $end]} {set i [nextLineStart $i]} {
	set end [pos::math $end -$increment]
	goto $i
	replaceText $i [pos::math $i + $increment] ""
    }
    goto $start
    select $start $end
}


#================================================================================
proc doM2ShiftRight {} {
    global M2RightShift
    if {![isSelection]} then {selectLine}
    set start [lineStart [getPos]]
    set end   [nextLineStart [pos::math [selEnd] -1]]
    select $start $end
    tabsToSpaces
    set increment [string length $M2RightShift]
    for {set i $start} {[pos::compare $i < $end]} {set i [nextLineStart $i]} {
	set end [pos::math $end + $increment]
	goto $i
	insertText $M2RightShift
    }
    goto $start
    select $start $end
}

#================================================================================
proc selectLine {} {
    set pos [getPos]
    set start [lineStart $pos]
    set end [nextLineStart $pos]
    select $start $end
}



#================================================================================
proc nextPlaceholder {} {
	searchPlaceholder 1
}
proc prevPlaceholder {} {
	searchPlaceholder 0
}


#================================================================================
proc searchPlaceholder {dir} {
    set pos [getPos]
    set depth 1
    if ($dir==1) {
	set push "(*."
	set pop  ".*)"
	if {[getSelect] != ""} {
	    set pos [pos::math $pos +1]
	}
	set add 3;
	set position [search -s -r 1 -f $dir -n -- "\\(\\*\\." $pos]
    } else {
	set push  ".*)"
	set pop   "(*."
	set pos [pos::math [selEnd] -4]
	set add -3;
	set position [search -s -r 1 -f $dir -n -- "\\.\\*\\)" $pos]
    }
    if {$position != ""} {
	set pos [pos::math [lindex $position 0] + $add]
	set str "(\\(\\*\\.)|(\\.\\*\\))"
	while {1} {
	    set limits [search -s -r 1 -f $dir -n -- "$str" $pos]
	    if {$limits == ""}  {
		message "Not matched placeholder"
		beep
		return
	    }
	    set pos [lindex $limits 0]
	    set c [getText $pos [pos::math $pos +3]]
	    if {$c == $push} {
		incr depth
	    } 
	    if {$c == $pop} {
		if {[set depth [expr $depth-1]] == 0} {
		    if ($dir==1) {
			select [lindex $position 0] [pos::math $pos + 3]
		    } else {
			select $pos [lindex $position 1]
		    }
		    return
		}
	    }
	    set pos [pos::math $pos + $add]
	    if {[pos::compare $pos > [maxPos]]} {
		alertnote "makro error, please contact jth"
	    }
	}
    } else {
	message "no more placeholders"
	beep
    }
}





# Reporting that end of this script has been reached
message "m2Edit.tcl for Programing in Modula-2 loaded"
if {$installDebugFlag} then {
	alertnote "m2Edit.tcl for Programing in Modula-2 loaded"
}
